#include "util/sync_control_parser.h"

#include "control/sync_control.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <glob.h>
#include <pwd.h>
#include <grp.h>

/* length of local list created to avoid malloc */
#define LOCAL_LIST_LEN 2
/* BLOCK_SEP_CHAR is used to seperate different blocks of configuration*/
#define BLOCK_SEP_CHAR ';'
/* ENTRY_SEP_CHAR is used to seperate different entries within a block */
#define ENTRY_SEP_CHAR ','
/* to ease reading a space can be helpful to arrange params */
#define WHITESPACE_CHAR ' '

#define ESCAPE_CHAR '\\'
#define MODE_BASE 8
#define MODE_MAX 0777
#define ABS_PATH_START_CHAR '/'
#define PATH_SEPARATOR "/"
#define SYSFS_PATH_START "/sys"
#define DEVFS_PATH_START "/dev"

//----------------------------- private function declaration ---------------------------------------------------------
static error_code_t sync_control_parser_parse_device_line_in_config(char *line);

static error_code_t sync_control_parser_parse_attribute_line_in_config(char *line);

static size_t sync_control_get_longest_string_from_list (char** list, size_t list_len);

static char* sync_control_str_split (char** str, char split_char,
		char escape_char);

static size_t sync_control_str_split_entry_cnt (const char* str, char split_char,
		char escape_char);

static void sync_control_remove_escaped_chars(char* str, char escape_char);

static error_code_t sync_control_check_unexpected_escape_sequences (
		const char* str, char escape_char, char split_char1,
		char split_char2, char whitespace_char);

static error_code_t sync_control_remove_whitespaces (char* str,
		char escape_char, char split_char1,
		char split_char2, char whitespace_char);

static error_code_t sync_control_parse_credentials_block (char* credentials_str,
		struct kms_credentials_t *credentials);

static bool sync_control_is_abs_path (const char *path);

static error_code_t sync_control_split_block_into_entries (char ***entry_list,
			size_t *entry_len, char *block);

static error_code_t sync_control_add_event_string_common(const char *event_string,
		const struct kms_credentials_t *credentials, size_t attr_str_list_len, char** const attr_str_list);

static error_code_t sync_control_process_stat_path_expansion_case_common(
		size_t stat_event_str_list_len, char** const stat_event_str_list,
		size_t dyn_event_str_list_len, char** const dyn_event_str_list,
		const struct kms_credentials_t *credential,
		size_t attr_str_list_len, char** const attr_str_list,
		char *copy_buffer);

static error_code_t sync_control_process_simple_case_common(
		size_t dyn_event_str_list_len, char** const dyn_event_str_list,
		const struct kms_credentials_t *credential,
		size_t attr_str_list_len, char** const attr_str_list);

static error_code_t sync_control_parse_line_common(char *line, bool is_attr);

//----------------------------- public attributes --------------------------------------------------------------------
const kml_conf_parser_t device_parser=
{
		.section_name="[kernel devices]",
		.parse_line=sync_control_parser_parse_device_line_in_config,
		.can_defere_parsing=true
};

const kml_conf_parser_t attribute_parser=
{
		.section_name="[kernel attributes]",
		.parse_line=sync_control_parser_parse_attribute_line_in_config,
		.can_defere_parsing=true
};
//--------------------------------------------------------------------------------------------------------------------

//----------------------------- private functions --------------------------------------------------------------------

static error_code_t sync_control_parser_parse_device_line_in_config(char *line)
{
	return sync_control_parse_line_common (line, false);
}

static error_code_t sync_control_parser_parse_attribute_line_in_config(char *line)
{
	return sync_control_parse_line_common (line, true);
}


/**
 * Returns the length of the longest string in an array of strings
 *
 * \param list list of strings
 * \param len amount of elements in the list of strings
 *
 * \return length of longest string within list.
 */
static size_t sync_control_get_longest_string_from_list (char** list, size_t list_len)
{
	size_t max_len = 0;
	size_t i=0;
	size_t tmp_len;

	if (list != NULL) {
		while (i<list_len) {
			if (list[i] != NULL) {
				tmp_len = strlen(list[i]);

				if (tmp_len > max_len)
					max_len = tmp_len;
			}

			i++;
		}
	}

	return max_len;
}


/**
 * split a string into chunks separated by split_char
 *
 * When the splitting point it found it will be replaced by \0
 * character and str pointer is updated
 *
 * split_char and escape_char need to be different
 *
 * Note : The escape character is not removed from the result
 * After the string is separated you need to call
 * sync_control_remove_escaped_chars in order to remove
 * escape characters
 *
 * \param str pointer to start of search string
 * 				result points to character following splitting point
 * 				or NULL if end of string has been reached
 * \param split_char character to be used for splitting the string in chunks
 * \param escape_char the character following this character will be ignored
 *		the only exception is a \0
 *
 * \return pointer to the string containing the portion up to split_char or
 * up to the end of str (\0).
 */
static char* sync_control_str_split (char** str, char split_char, char escape_char)
{
	char* res = *str;
	if (res != NULL) {
		while (**str != '\0') {
			if (**str == escape_char) {
				*str += 1;
				/* did the escape char occur directly in front of end of string ? */
				if (**str == '\0')
					break;
			} else {
				/* found split_char not preceeded by escape_char */
				if (**str == split_char) {
					**str = '\0';
					*str += 1;
					return res;
				}
			}
			*str += 1;
		}
		*str = NULL;
	}

	return res;
}

/**
 * count the number of elements which would be found when calling
 * sync_control_str_split repeatedly
 *
 * \param str pointer to start of search string
 * \param split_char character to be used for splitting the string in chunks
 * \param escape_char the character following this character will be ignored
 *		the only exception is a \0
 *
 * \return number of entries sync_control_str_split would return when
 * 		calling sync_control_str_split repeatedly
 */
static size_t sync_control_str_split_entry_cnt (const char* str, char split_char, char escape_char)
{
	size_t cnt = 0;

	if (str != NULL) {
		cnt=1;

		while (*str != '\0') {
			if (*str == escape_char) {
				str++;
				/* did the escape char occur directly in front of end of string ? */
				if (*str == '\0')
					break;
			} else {
				/* found split_char not preceeded by escape_char */
				if (*str == split_char) {
					cnt++;
				}
			}
			str++;
		}
	}

	return cnt;
}

/**
 * remove escape_characters from the string
 *
 * sequence (escape_char, any char) is replaced by simply (any_char)
 *
 * \param str pointer to start of string
 * \param escape_char the character used for escaping the following character
 *
 */
static void sync_control_remove_escaped_chars(char* str, char escape_char)
{
	char* read_ptr;
	char* write_ptr;

	read_ptr = str;
	if (read_ptr != NULL) {
		write_ptr = read_ptr;
		while (*read_ptr != '\0') {
			if (*read_ptr == escape_char) {
				read_ptr++;

				if (*read_ptr == '\0')
					break;

				*write_ptr = *read_ptr;
			} else {
				*write_ptr = *read_ptr;
			}
			read_ptr++;
			write_ptr++;
		}
		*write_ptr = '\0';
	}
}

/**
 * Check if any unexpected escape sequence occurs in the string
 *
 * \param str pointer to start of search string
 * \param escape_char character to start a escape sequence
 * \param split_char1 first character to be used for splitting the string
 * \param split_char2 second character to be used for splitting the string
 * \param whitespace_char whitespace character
 *
 * \return RESULT_OK if no invalid escape sequence was detected
 * 			RESULT_CONF_INVALID in case of invalid escape sequence
 */
static error_code_t sync_control_check_unexpected_escape_sequences (
		const char* str, char escape_char, char split_char1,
		char split_char2, char whitespace_char) {
	const char *ptr = str;
	while (*ptr != '\0') {
		if (*ptr == escape_char) {
			ptr++;
			if ( (*ptr != escape_char ) &&
				 (*ptr != split_char1 ) &&
				 (*ptr != split_char2 ) &&
				 (*ptr != whitespace_char ) )
			{
				logger_log_debug("SYNC_CONTROL_PARSER -> \"%s\" contains invalid escape sequence \"%c%c\"",
							str, escape_char, *ptr);
				return RESULT_CONF_INVALID;
			}
		}
		if (*ptr != '\0') {
			ptr++;
		}
	}
	return RESULT_OK;
}


/**
 * Remove the whitespace character from the string
 * Escaped whitespace characters will be ignored
 * Whitespaces are only allowed before or after
 * characters for splitting
 *
 * \param str pointer to start of search string
 * \param whitespace_char whitespace character
 * \param split_char1 first character to be used for splitting the string
 * \param split_char2 second character to be used for splitting the string
 * \param escape_char character to start a escape sequence
 *
 * \return RESULT_OK everything is OK
 * 		RESULT_WHITESPACE_INVALID in case of invalid position of whitespaces
 */
static error_code_t sync_control_remove_whitespaces (char* str,
		char escape_char, char split_char1,
		char split_char2, char whitespace_char) {
	error_code_t result = RESULT_OK;
	char* read_ptr;
	char* write_ptr;
	/* enum for tracking state in sync_control_remove_whitespaces */
	enum remove_whitespace_state { NO_WHITESPACE, AFTER_SPLIT,
		WHITESPACE_POTENTIALLY_BEFORE_SPLIT } state = AFTER_SPLIT;

	read_ptr = str;
	if (read_ptr != NULL) {
		write_ptr = read_ptr;
		while (*read_ptr != '\0') {
			if (*read_ptr == escape_char) {
				if (state == WHITESPACE_POTENTIALLY_BEFORE_SPLIT)
					result = RESULT_WHITESPACE_INVALID;

				state = NO_WHITESPACE;

				*write_ptr = *read_ptr;
				read_ptr++;
				write_ptr++;

				if (*read_ptr == '\0')
					break;

				*write_ptr = *read_ptr;
			} else {
				if (*read_ptr == whitespace_char) {
					if (state != AFTER_SPLIT)
						state = WHITESPACE_POTENTIALLY_BEFORE_SPLIT;

					/* don't copy whitespace - just move read_ptr*/
					read_ptr++;
					continue;
				} else {
					if ((*read_ptr == split_char1) ||
						(*read_ptr == split_char2)) {
						state = AFTER_SPLIT;
					} else {
						if (state == WHITESPACE_POTENTIALLY_BEFORE_SPLIT)
							result = RESULT_WHITESPACE_INVALID;

						state = NO_WHITESPACE;
					}
					/* copy character from read_ptr to write_ptr */
					*write_ptr = *read_ptr;
				}
			}
			read_ptr++;
			write_ptr++;
		}
		*write_ptr = '\0';
	}

	if (result != RESULT_OK)
		logger_log_error("Whitespace not in front or after splitting characters.");

	return result;
}

/**
 * parse the configuration block for configuring rights to be set
 *
 * \param credentials_str pointer to start of string containing
 * 				user, group, mode (might be NULL if no rights need to be set)
 * \param credentials struct to return results - must not be NULL
 *
 * \return RESULT_OK if valid credentials are found
 * 			RESULT_CREDENTIALS_EMPTY if no credentials have been configured
 * 			RESULT_CREDENTIALS_INVALID in case of invalid configuration
 */
static error_code_t sync_control_parse_credentials_block (char* credentials_str,
		struct kms_credentials_t *credentials)
{
	char* user;
	char* group;
	char* mode;
	char* pos = credentials_str;
	struct passwd *userinfo;
	struct group *groupinfo;
	long tmp_mode;

	credentials->user_id = 0;
	credentials->group_id = 0;
	credentials->mode = 0;

	if ((credentials_str == NULL) || (*credentials_str == '\0'))
		return RESULT_CREDENTIALS_EMPTY;

	logger_log_debug("SYNC_CONTROL_PARSER -> evaluate credential string \"%s\".",
			credentials_str);

	user = sync_control_str_split (&pos, ENTRY_SEP_CHAR, ESCAPE_CHAR);
	group = sync_control_str_split (&pos, ENTRY_SEP_CHAR, ESCAPE_CHAR);
	mode = sync_control_str_split (&pos, ENTRY_SEP_CHAR, ESCAPE_CHAR);

	if ((pos != NULL) ||
		(user == NULL) ||
		(group == NULL) ||
		(mode == NULL)) {
		logger_log_error("Invalid credentials string.");
		return RESULT_CREDENTIALS_INVALID;
	}

	sync_control_remove_escaped_chars (user, ESCAPE_CHAR);
	sync_control_remove_escaped_chars (group, ESCAPE_CHAR);
	sync_control_remove_escaped_chars (mode, ESCAPE_CHAR);

	userinfo = getpwnam(user);
	if (userinfo != NULL) {
		credentials->user_id = userinfo->pw_uid;
	} else {
		logger_log_error("User (\"%s\") not found.", user);
		return RESULT_CREDENTIALS_INVALID;
	}

	groupinfo = getgrnam(group);
	if (groupinfo != NULL) {
		credentials->group_id = groupinfo->gr_gid;
	} else {
		logger_log_error("Group (\"%s\") not found.", group);
		return RESULT_CREDENTIALS_INVALID;
	}

	tmp_mode = strtol (mode, &pos, MODE_BASE);
	credentials->mode = (mode_t)tmp_mode;
	/* check if mode has been processed completely */
	if ((*pos != '\0') || (tmp_mode > MODE_MAX)) {
		logger_log_error("Invalid mode (\"%s\") configuration.", mode);
		return RESULT_CREDENTIALS_INVALID;
	}

	return RESULT_OK;
}

/**
 * simply check if path is starting with '/'
 *
 * \param path string to be checked - must not be called with NULL ptr
 *
 * \return true if first char is '/', false otherwise
 */
static bool sync_control_is_abs_path (const char *path) {
	return *path == ABS_PATH_START_CHAR;
}

/**
 * Split a block into multiple entries
 * The result is stored to entry_list, the number of found entries is returned
 * using entry_len
 * Initially entry_len specifies the amount of storage available for entry_list
 * (created externally)
 * In case this amount is not sufficient malloc is used to allocate a sufficient
 * memory area. It is the responsibility of the function calling this function
 * to free this memory after usage.
 * This is useful to avoid malloc in the common case.
 *
 * \param entry_list pointer to array of string pointers
 * 		when calling the pointer to external available array of string can be given
 * 		when returning it will either point to the same array or in case the memory
 * 		was not sufficient to the memory area allocated to store the result
 * \param entry_len pointer to length of array
 * 		when calling this pointer is used to specify the amount of storage available
 * 		for external array of strings (use 0 in case no memory is available externally)
 * 		when returning it will be used to return the amount of entries found in the
 * 		parsed block
 * \param block the block to be split into entries
 *
 * \return RESULT_OK in case of no error
 * 			RESULT_NORESOURCES in case no memory was available to store the results
 */
static error_code_t sync_control_split_block_into_entries (char ***entry_list,
			size_t *entry_len, char *block) {

	error_code_t result = RESULT_OK;
	size_t entries_in_block;
	size_t i;
	char *ptr = block;
	char **str_list;

	entries_in_block = sync_control_str_split_entry_cnt (ptr,
			ENTRY_SEP_CHAR, ESCAPE_CHAR);

	if (entries_in_block > *entry_len) {
		/* passed storage is not sufficient, malloc storage */
		void* listmem = calloc (sizeof(char*), entries_in_block);
		*entry_list = listmem;
	}

	str_list = *entry_list;

	if (!str_list)
		return RESULT_NORESOURCES;

	*entry_len = 0;
	i = 0;
	while (ptr != NULL) {
		str_list[i] = sync_control_str_split (&ptr,
				ENTRY_SEP_CHAR, ESCAPE_CHAR);

		if (str_list[i] != NULL) {
			sync_control_remove_escaped_chars(str_list[i],
					 ESCAPE_CHAR);
		}

		i++;
		*entry_len = i;
	}

	return result;
}


/**
 * check if the event string belongs to /sys or
 * /dev (only for device event strings) and afterwards
 * add it to the corresponding lists
 *
 * \param event_string sync_point (must not be NULL)
 * \param credentials credentials set or
 * 		NULL if no credentials shall be set
 * \param attr_str_list_len amount of entries in attr_str_list
 * 		set to 0 in device parsing mode
 * \param attr_str_list list of elements to be processed
 * 		set to NULL in device parsing mode
 *
 * \return RESULT_OK if OK, error_code otherwise
 */
static error_code_t sync_control_add_event_string_common(const char *event_string,
		const struct kms_credentials_t *credentials, size_t attr_str_list_len, char** const attr_str_list) {
	error_code_t result=RESULT_OK;
	size_t i;

	logger_log_debug("SYNC_CONTROL_PARSER -> add device or attributes for path %s - attribute list length is %d.",
			event_string, attr_str_list_len);

	if (attr_str_list_len > 0) {
		/* attribute parsing case */
		if (strncmp(event_string, SYSFS_PATH_START,
					strlen(SYSFS_PATH_START)) != 0) {
			logger_log_error("Event string \"%s\" does not belong to sysfs.",
					event_string);
			result = RESULT_EVENT_STRING_INVALID;
		} else {

			i=0;
			while ((result == RESULT_OK) && (i < attr_str_list_len))
			{
				if (strlen(attr_str_list[i]) != 0) {

					if (sync_control_is_abs_path(attr_str_list[i]) == true) {
						logger_log_error("Attributes need to be relative (%s).",
								attr_str_list[i]);
						result = RESULT_ATTRIBUTE_INVALID;
					} else {
						result = sync_control_add_attribute(event_string,
							attr_str_list[i] , credentials);
					}
				} else {
					logger_log_error("Zero length attribute not supported.");
					result = RESULT_CONF_INVALID;
				}

				i++;
			}
		}
	} else {
		/* device parsing case */
		if ((strncmp(event_string, SYSFS_PATH_START,
					strlen(SYSFS_PATH_START)) != 0) &&
			(strncmp(event_string, DEVFS_PATH_START,
					strlen(DEVFS_PATH_START)) != 0)) {
			logger_log_error("Event string \"%s\" does not belong to devfs or sysfs.",
					event_string);
			result = RESULT_EVENT_STRING_INVALID;
		} else {
			result = sync_control_add_device(event_string, credentials);
		}
	}

	return result;
}

/**
 * The function sync_control_parse_line_common will call this function
 * to handle configuration lines with static event string path.
 * This function will combine each stat_event_str from stat_event_str_list
 * with each dyn_event_str_list from dyn_event_str before calling
 * sync_control_add_event_string_common for further processing
 *
 * \param stat_event_str_list_len amount of entries in stat_event_str_list
 * \param stat_event_str_list list of elements to be processed
 * \param dyn_event_str_list_len amount of entries in dyn_event_str_list
 * \param dyn_event_str_list list of elements to be processed
 * \param credentials credentials set or
 * 		NULL if no credentials shall be set
 * \param attr_str_list_len amount of entries in attr_str_list
 * 		set to 0 in device parsing mode
 * \param attr_str_list list of elements to be processed
 * 		set to NULL in device parsing mode
 * \param copy_buffer the buffer used for concatenation
 * 		size has to be sufficient to allow concatenation of any
 * 		stat_event_str and any dyn_event_string (including / and \0)
 * 		must not be NULL
 *
 * \return RESULT_OK if OK, error_code otherwise
 */
static error_code_t sync_control_process_stat_path_expansion_case_common(
		size_t stat_event_str_list_len, char** const stat_event_str_list,
		size_t dyn_event_str_list_len, char** const dyn_event_str_list,
		const struct kms_credentials_t *credential,
		size_t attr_str_list_len, char** const attr_str_list,
		char *copy_buffer) {

	error_code_t result=RESULT_OK;
	size_t stat_list_iter;
	size_t dyn_list_iter;
	char *dyn_pos_in_copy_buffer;

	logger_log_debug ("SYNC_CONTROL_PARSER -> sync_control_process_stat_path_expansion_case_common processing a list of %d dynamic event strings and %d paths matching to static event string.",
			dyn_event_str_list_len, stat_event_str_list_len);

	/* check dynamic event strings one time */
	dyn_list_iter=0;
	while ((result == RESULT_OK) && (dyn_list_iter < dyn_event_str_list_len))
	{
		if (strlen(dyn_event_str_list[dyn_list_iter]) == 0) {
			logger_log_error("Zero length dynamic event string not supported.");
				result = RESULT_CONF_INVALID;
		}

		dyn_list_iter++;
	}

	stat_list_iter = 0;
	while ((result == RESULT_OK) && (stat_list_iter < stat_event_str_list_len)) {
		/* replace content of copy buffer by current static event path and append / */
		strcpy(copy_buffer, stat_event_str_list[stat_list_iter]);
		strcat(copy_buffer, PATH_SEPARATOR);
		/* keep position to start appending dynamic event string there */
		dyn_pos_in_copy_buffer = copy_buffer + strlen(copy_buffer);

		dyn_list_iter=0;
		while ((result == RESULT_OK) && (dyn_list_iter < dyn_event_str_list_len))
		{
			if (sync_control_is_abs_path(dyn_event_str_list[dyn_list_iter]) == true) {

				/* call with sysfs path being absolute dynamic event string */
				result = sync_control_add_event_string_common (
									dyn_event_str_list[dyn_list_iter],
									credential,
									attr_str_list_len,
									attr_str_list);

			} else {
				/* append dynamic event string */
				strcpy(dyn_pos_in_copy_buffer, dyn_event_str_list[dyn_list_iter]);

				/* call with sysfs path being the combination of matching
				 * static event path and relative dynamic event path
				 */
				result = sync_control_add_event_string_common (
									copy_buffer,
									credential,
									attr_str_list_len,
									attr_str_list);
			}

			dyn_list_iter++;
		}

		stat_list_iter++;
	}

	return result;
}


/**
 * The function sync_control_parse_line_common will call this function
 * to handle configuration lines without static event string path.
 * This function will iterate over all dynamic event strings listed in
 * dyn_event_str_list call sync_control_add_event_string_common for
 * further processing
 *
 * \param dyn_event_str_list_len amount of entries in dyn_event_str_list
 * \param dyn_event_str_list list of elements to be processed
 * \param credentials credentials set or
 * 		NULL if no credentials shall be set
 * \param attr_str_list_len amount of entries in attr_str_list
 * 		set to 0 in device parsing mode
 * \param attr_str_list list of elements to be processed
 * 		set to NULL in device parsing mode
 *
 * \return RESULT_OK if OK, error_code otherwise
 */
static error_code_t sync_control_process_simple_case_common(
		size_t dyn_event_str_list_len, char** const dyn_event_str_list,
		const struct kms_credentials_t *credential,
		size_t attr_str_list_len, char** const attr_str_list) {
	error_code_t result=RESULT_OK;
	size_t iter;

	logger_log_debug ("SYNC_CONTROL_PARSER -> sync_control_process_simple_case_common processing a list of %d dynamic event strings.",
			dyn_event_str_list_len);

	iter=0;
	while ((result == RESULT_OK) && (iter < dyn_event_str_list_len))
	{
		if (strlen(dyn_event_str_list[iter]) != 0) {
			result = sync_control_add_event_string_common (
					dyn_event_str_list[iter],
					credential,
					attr_str_list_len,
					attr_str_list);
		} else {
			logger_log_error("Zero length dynamic event string not supported.");
				result = RESULT_CONF_INVALID;
		}

		iter++;
	}

	return result;
}


/**
 * Common function to handle device_sync and attribute configuration lines
 * for devices the syntax is dynamic_sysfs_paths[;credentials[;static_sysfs_path]]
 * for attributes the syntax is dynamic_sysfs_paths;attribute_names[;credentials[;static_sysfs_path]]
 *
 * static_sysfs_path is a sysfs path always present - device instances will be created as sub-paths
 * 		These folders exist even if the device driver has not been probed
 * 		In general this is a sysfs entry belonging to a DT entry specifying that a certain device
 * 		of a certain class shall be probed
 * 		This path can contain wildcards
 * 		When parsing the configuration line it is checked if one or multiple paths are matching
 * 		For each matching path the final sysfs path is generated by appending the relativ
 * 		dynamic sysfs path
 * credentials - user,group,mode (e.g. root,daemon,775)
 * attribute_names = attribute_name[,attribute_names]
 * 		This is the name of one or multiple attributes
 * 		Only relevant for configuration lines belonging to attributes
 * dynamic_sysfs_paths = dynamic_sysfs_path[:dynamic_sysfs_paths]
 * 		This block contains one or multiple sysfs paths
 * 		These paths can be absolute or relative
 * 		Relative paths are only permitted in combination with a static_sysfs_path
 * 		Relative paths will be concatenated with the paths matching static_sysfs_path pattern
 * 		In case of a device the path can refer to sysfs or devfs
 * 		In case of an attribute only sysfs paths are permitted
 *
 * To improve readability it is allowed to put whitespaces before or after characters used to
 * separate the configuration items (i.e. "," and ";")
 * ",", ";" and " " might be used in path if escaped by "\"
 *
 * \param line the configuration line
 * \param is_attr is this an attribute or a device_sync sync_point
 *
 * \return RESULT_OK if OK, error_code otherwise
 */
static error_code_t sync_control_parse_line_common(char *line, bool is_attr)
{
	error_code_t result;

	char* str_pos = line;
	char *dyn_event_str_block;
	char *attributes_block = NULL;
	char *credentials_block;
	char *stat_event_str_block;
	char *stat_event_str;

	size_t dyn_event_str_list_len = LOCAL_LIST_LEN;
	/* local list avoid malloc for the common case */
	#if LOCAL_LIST_LEN > 0
		char *dyn_event_str_list_local[LOCAL_LIST_LEN];
	#endif
	char **dyn_event_str_list;

	size_t attr_str_list_len = LOCAL_LIST_LEN;
	/* local list avoid malloc for the common case */
	#if LOCAL_LIST_LEN > 0
		char *attr_str_list_local[LOCAL_LIST_LEN];
	#endif
	char **attr_str_list = NULL;

	struct kms_credentials_t tmp_credentials;
	struct kms_credentials_t *credentials = NULL;

	size_t copybuffer_len;
	char *copybuffer;

	int gl_result;
	glob_t gl;

	logger_log_debug("SYNC_CONTROL_PARSER -> Parsing %s line : %s",
			is_attr ? "attribute" : "device_sync",
			line);

	result = sync_control_check_unexpected_escape_sequences (
			str_pos, ESCAPE_CHAR, BLOCK_SEP_CHAR, ENTRY_SEP_CHAR, WHITESPACE_CHAR);
	if (result != RESULT_OK)
		return result;

	result = sync_control_remove_whitespaces (
			str_pos, ESCAPE_CHAR, BLOCK_SEP_CHAR, ENTRY_SEP_CHAR, WHITESPACE_CHAR);
	if (result != RESULT_OK)
			return result;

	/* split the line into the configuration blocks */
	dyn_event_str_block = sync_control_str_split (&str_pos,
			BLOCK_SEP_CHAR, ESCAPE_CHAR);
	if (is_attr)
		attributes_block = sync_control_str_split (&str_pos,
				BLOCK_SEP_CHAR, ESCAPE_CHAR);
	credentials_block = sync_control_str_split (&str_pos,
			BLOCK_SEP_CHAR, ESCAPE_CHAR);
	stat_event_str_block = sync_control_str_split (&str_pos,
			BLOCK_SEP_CHAR, ESCAPE_CHAR);

	/* check that line ends after static event string block */
	if (str_pos != NULL) {
		logger_log_error("Unexpected line end.");
		return RESULT_CONF_INVALID;
	}

	stat_event_str = sync_control_str_split (&stat_event_str_block,
			ENTRY_SEP_CHAR, ESCAPE_CHAR);
	/* check that there is only one static event string configured */
	if (stat_event_str_block != NULL) {
		logger_log_error("Invalid static event string configuration. Only one static event string is supported.");
		return RESULT_CONF_INVALID;
	}

	/* check that at least one dynamic event string block is configured */
	if ((dyn_event_str_block == NULL) || (strlen(dyn_event_str_block) == 0)) {
		logger_log_error("Invalid dynamic event string configuration. Block is empty.");
		return RESULT_CONF_INVALID;
	}

	if (is_attr) {
		/* check that at least one attribute is configured */
		if ((attributes_block == NULL) || (strlen(attributes_block) == 0)) {
			logger_log_error("Invalid attribute configuration. Block is empty.");
			return RESULT_CONF_INVALID;
		}
	}

	#if LOCAL_LIST_LEN > 0
		dyn_event_str_list = &dyn_event_str_list_local[0];
	#else
		dyn_event_str_list = NULL;
	#endif
	result = sync_control_split_block_into_entries (&dyn_event_str_list,
			&dyn_event_str_list_len, dyn_event_str_block);

	if (result == RESULT_OK) {
		if (is_attr) {
			#if LOCAL_LIST_LEN > 0
				attr_str_list = &attr_str_list_local[0];
			#else
				attr_str_list = NULL;
			#endif
			result = sync_control_split_block_into_entries (&attr_str_list,
					&attr_str_list_len, attributes_block);
		} else {
			/* below this point (attr_str_list==NULL) is used to mark the
			 * device parsing case
			 */
			attr_str_list_len = 0;
			attr_str_list = NULL;
		}
	}

	if (result == RESULT_OK) {
		if ((stat_event_str != NULL) && (strlen(stat_event_str) != 0))
		{
			/* in case of a static event string part configured we need to
			 * use glob to lookup all matching paths in SYSFS
			 */
			sync_control_remove_escaped_chars (stat_event_str, ESCAPE_CHAR);

			/* check if it is a sysfs path */
			if (strncmp(stat_event_str, SYSFS_PATH_START,
					strlen(SYSFS_PATH_START))) {
				logger_log_error("Invalid static event string configuration. Path \"%s\" is no sysfs path.",
						stat_event_str);
				return RESULT_EVENT_STRING_INVALID;
			}

			/* search for all paths matching  with wildcards */
			gl_result=glob (stat_event_str, GLOB_NOSORT | GLOB_ERR, NULL, &gl);
			logger_log_debug("SYNC_CONTROL_PARSER -> Checking static event string path %s: gl_result %d, %d matching paths",
					stat_event_str, gl_result, gl.gl_pathc);
			if (gl_result==0)
			{
				/* we have at least one match
				 * so we should resolve the rights configuration now */
				result = sync_control_parse_credentials_block (credentials_block,
						&tmp_credentials);

				if ((result == RESULT_OK) || (result == RESULT_CREDENTIALS_EMPTY)) {
					/* in case credentials configuration is not empty we need to pass
					 * pointer to tmp_credentials struct when creating entries
					 * if not we will pass NULL = no credentials
					 */
					if (result == RESULT_OK)
						credentials = &tmp_credentials;

					/* determine the length of the buffer needed for concatenation */
					copybuffer_len = 2;
					copybuffer_len += sync_control_get_longest_string_from_list(
							gl.gl_pathv, gl.gl_pathc);

					/* this is just an estimate, as not all dynamic parts will be defined
					 * relative to matching static part - but a good estimate as typically
					 * relative paths are used
					 */
					copybuffer_len += sync_control_get_longest_string_from_list(
							dyn_event_str_list, dyn_event_str_list_len);

					copybuffer = calloc (sizeof(char), copybuffer_len);

					if (copybuffer != NULL) {
						result = sync_control_process_stat_path_expansion_case_common (
								gl.gl_pathc, gl.gl_pathv,
								dyn_event_str_list_len, dyn_event_str_list,
								credentials,
								attr_str_list_len, attr_str_list,
								copybuffer);

						free(copybuffer);
					} else {
						result = RESULT_NORESOURCES;
					}
				}
			} else {
				if (gl_result != GLOB_NOMATCH)
					result = RESULT_NORESOURCES;
			}
			globfree(&gl);
		} else {
			/* simple case
			 * so we should resolve the rights configuration now */
			result = sync_control_parse_credentials_block (credentials_block,
					&tmp_credentials);

			if ((result == RESULT_OK) || (result == RESULT_CREDENTIALS_EMPTY)) {
				/* in case credentials configuration is not empty we need to pass
				 * pointer to tmp_credentials struct when creating entries
				 * if not we will pass NULL = no credentials
				 */
				if (result == RESULT_OK)
					credentials = &tmp_credentials;

				result = sync_control_process_simple_case_common (
						dyn_event_str_list_len, dyn_event_str_list,
						credentials,
						attr_str_list_len, attr_str_list);
			}
		}
	}

	/* if sync_control_split_block_into_entries has allocated memory to store
	 * results we need to free it
	 */
	if ((dyn_event_str_list != &dyn_event_str_list_local[0]) && (dyn_event_str_list != NULL))
		free(dyn_event_str_list);
	if ((attr_str_list != &attr_str_list_local[0]) && (attr_str_list != NULL))
		free(attr_str_list);

	return result;
}
//--------------------------------------------------------------------------------------------------------------------
